package loquebot.body;

import java.util.Iterator;
import java.util.ArrayList;
import java.util.logging.Logger;

import cz.cuni.pogamut.Client.AgentBody;

import cz.cuni.pogamut.MessageObjects.MessageObject;
import cz.cuni.pogamut.MessageObjects.MessageType;
import cz.cuni.pogamut.MessageObjects.Triple;

import cz.cuni.pogamut.MessageObjects.NavPoint;
import cz.cuni.pogamut.MessageObjects.Reachable;

import loquebot.Main;
import loquebot.util.LoqueListener;
import loquebot.util.LoqueRequestID;
import loquebot.memory.LoqueMemory;

/**
 * Responsible for navigation to location.
 *
 * <p>This class navigates the agent along path. Silently handles all casual
 * trouble with preparing next nodes, running along current nodes, switching
 * between nodes at appropriate distances, etc. In other words, give me a
 * destination and a path and you'll be there in no time.</p>
 *
 * <h4>Preparing ahead</h4>
 *
 * Nodes in the path are being prepared ahead, even before they are actually
 * needed. The agent decides ahead, looks at the next nodes while still running
 * to current ones, etc.
 *
 * <h4>Reachability checks</h4>
 *
 * Whenever the agent switches to the next node, reachcheck request is made to
 * the engine. The navigation routine then informs the {@link LoqueRunner}
 * beneath about possible troubles along the way.
 *
 * <h4>Movers</h4>
 *
 * This class was originally supposed to contain handy (and fully working)
 * navigation routines, including safe navigation along movers. However, the
 * pogamut platform is not quite ready for movers yet. Especial when it comes
 * to mover frames and correct mover links.
 *
 * <p>Thus, we rely completely on navigation points. Since the mover navigation
 * points (LiftCenter ones) always travel with the associated mover, we do not
 * try to look for movers at all. We simply compare navigation point location
 * to agent's location and wait or move accordingly.</p>
 *
 * <h4>Future</h4>
 *
 * The bot could check from time to time, whether the target destination he is
 * traveling to is not an empty pickup spot, since the memory is now capable of
 * reporting empty pickups, when they are visible. The only pitfall to this is
 * the way the agent might get <i>trapped</i> between two not-so-far-away items,
 * each of them empty. The more players playe the same map, the bigger is the
 * chance of pickup emptyness. The agent should implement a <i>fadeing memory
 * of which items are empty</i> before this can be put safely into logic.
 *
 * @author Juraj Simlovic [jsimlo@matfyz.cz]
 * @version Tested on Pogamut 2 platform version 1.0.5.
 */
public class LoqueNavigator
{
    /**
     * Current navigation destination.
     */
    private Triple navigDestination = null;

    /**
     * Current stage of the navigation.
     */
    private Stage navigStage = Stage.COMPLETED;

    /**
     * Traveling timeout..
     */
    private int navigTimeout = 0;

    /*========================================================================*/

    /**
     * Distance, which is considered as close enough..
     */
    public static final int CLOSE_ENOUGH = 60;

    /*========================================================================*/

    /**
     * Initializes direct navigation to the specified destination.
     * @param dest Destination of the navigation.
     * @param timeout Maximum timeout of the navigation. Use 0 to auto-timeout.
     */
    public void initDirectNavigation (Triple dest, int timeout)
    {
        // calculate destination distance
        int distance = (int) memory.self.getSpaceDistance (dest);
        // setup travel timeout based on distance
        timeout = getTimeout (timeout, distance);
        // init the navigation
        log.fine (
            "Navigator.initDirectNavigation(): initializing direct navigation"
            + ", distance " + distance
            + ", timeout " + timeout
        );
        // setup navigation timeout
        navigTimeout = timeout;
        // init direct navigation
        initDirectly (dest);
    }

    /**
     * Initializes navigation to the specified destination along specified path.
     * @param dest Destination of the navigation.
     * @param path Navigation path to the destination.
     * @param timeout Maximum timeout of the navigation.
     */
    public void initPathNavigation (Triple dest, ArrayList<NavPoint> path, int timeout)
    {
        // do not allow two jumping sequences
        if (timeout <= 0)
            throw new RuntimeException ("no timeout specified");

        // init the navigation
        log.fine (
            "Navigator.initPathNavigation(): initializing path navigation"
            + ", nodes " + path.size ()
            + ", timeout " + timeout
        );
        // setup navigation timeout
        navigTimeout = timeout;
        // init path navigation
        if (!initAlongPath(dest, path))
        {
            // do it directly then..
            initDirectNavigation (dest, timeout);
        }
    }

    /*========================================================================*/

    /**
     * Navigates with the current navigation request.
     * @return Stage of the navigation progress.
     */
    public Stage keepNavigating ()
    {
        // is there any point in navigating further?
        if (navigStage.terminated)
            return navigStage;

        // are we forced from above?
        if (main._fixFailNavigation)
        {
            log.fine ("Navigator.keepNavigating(): navigation canceled");
            return navigStage = Stage.CANCELED;
        }

        // try to navigate
        switch (navigStage)
        {
            case REACHING:
                navigStage = navigDirectly ();
                break;
            default:
                navigStage = navigAlongPath ();
                break;
        }

        // return the stage
        log.finest ("Navigator.keepNavigating(): navigation stage " + navigStage);
        return navigStage;
    }

    /*========================================================================*/

    /**
     * Initializes direct navigation to given destination.
     * @param dest Destination of the navigation.
     * @return Next stage of the navigation progress.
     */
    private Stage initDirectly (Triple dest)
    {
        // setup navigation info
        navigDestination = dest;
        // init runner
        runner.initRunner ();
        // reset navigation stage
        return navigStage = Stage.REACHING;
    }

    /**
     * Tries to navigate the agent directly to the navig destination.
     * @return Next stage of the navigation progress.
     */
    private Stage navigDirectly ()
    {
        // get the distance from the target
        int distance = (int) memory.self.getSpaceDistance (navigDestination);

        // are we there yet?
        if (distance <= CLOSE_ENOUGH)
        {
            log.fine ("Navigator.navigDirectly(): destination close enough: " + distance);
            return Stage.COMPLETED;
        }

        // is it taking too long?
        if (navigTimeout-- <= 0)
        {
            log.fine ("Navigator.navigDirectly(): navigation timeout");
            return Stage.TIMEOUT;
        }

        // run to that location..
        if (!runner.runToLocation (navigDestination, navigDestination, true))
        {
            log.fine ("Navigator.navigDirectly(): direct navigation failed");
            return Stage.CRASHED;
        }

        // well, we're still running
        log.finest ("Navigator.navigDirectly(): traveling directly: " + distance);
        return navigStage;
    }

    /*========================================================================*/

    /**
     * Iterator through navigation path.
     */
    private Iterator<NavPoint> navigIterator = null;

    /**
     * Last node in the path (the one the agent aready reached).
     */
    private NavPoint navigLastNode = null;

    /**
     * Current node in the path (the one the agent is running to).
     */
    private NavPoint navigCurrentNode = null;

    /**
     * Next node in the path (the one being prepared).
     */
    private NavPoint navigNextNode = null;

    /**
     * Node navigation timeout..
     */
    private int navigNodeTimeout = 0;

    /**
     * Initializes navigation along path.
     * @param dest Destination of the navigation.
     * @param path Path of the navigation.
     * @return True, if the navigation is successfuly initialized.
     */
    private boolean initAlongPath (Triple dest, ArrayList<NavPoint> path)
    {
        // setup navigation info
        navigDestination = dest;
        navigIterator = path.iterator ();
        // reset current node
        navigCurrentNode = null;
        // prepare next node
        prepareNextNode ();
        // reset navigation stage
        navigStage = Stage.NAVIGATING;
        // reset node navigation info
        return switchToNextNode ();
    }

    /**
     * Tries to navigate the agent safely along the navigation path.
     * @return Next stage of the navigation progress.
     */
    private Stage navigAlongPath ()
    {
        // get the distance from the destination
        int totalDistance = (int) memory.self.getSpaceDistance (navigDestination);

        // are we there yet?
        if (totalDistance <= CLOSE_ENOUGH)
        {
            log.finest ("Navigator.navigAlongPath(): destination close enough: " + totalDistance);
            return Stage.COMPLETED;
        }

        // is it taking too long?
        if (navigTimeout-- <= 0)
        {
            log.fine ("Navigator.navigAlongPath(): navigation timeout");
            return Stage.TIMEOUT;
        }

        // is it taking too long?
        if (navigNodeTimeout-- <= 0)
        {
            log.fine ("Navigator.navigAlongNodes(): current node timeout");
            return Stage.TIMEOUT;
        }

        // navigate
        return navigStage.mover
            ? navigThroughMover ()
            : navigToCurrentNode ();
    }

    /*========================================================================*/

    /**
     * Last reachcheck request ID.
     */
    private LoqueRequestID navigPrepareRequest = new LoqueRequestID ();

    /**
     * Prepares next navigation node in path.
     */
    private void prepareNextNode ()
    {
        // synchronize with listener
        synchronized (navigPrepareRequest)
        {
            // retreive the next node, if there are any left
            // note: there might be null nodes along the path!
            navigNextNode = null;
            while ((navigNextNode == null) && navigIterator.hasNext ())
            {
                // get next node in the path
                navigNextNode = navigIterator.next ();
            }

            // did we get the next node?
            if (navigNextNode == null)
            {
                // clear the old request id.. since we set the navigNextNode
                // to null, response listener should not try to update it
                navigPrepareRequest.ID = -1;
                return;
            }

            // request reachcheck to the next node..
            log.finer ("Navigator.prepareNextNode(): preparing next node " + navigNextNode.UnrealID);
            body.requestReachcheckLocation (
                navigPrepareRequest.createNewID(1), navigNextNode.location,
                (navigCurrentNode == null) ? memory.self.getLocation () : navigCurrentNode.location
            );
        }
        return;
    }

    /**
     * Initializes next navigation node in path.
     * @return True, if the navigation node is successfuly switched.
     */
    private boolean switchToNextNode ()
    {
        // move the current node into last node
        navigLastNode = navigCurrentNode;

        // get the next prepared node
        if (null == (navigCurrentNode = navigNextNode))
        {
            // no nodes left there..
            log.finer ("Navigator.switchToNextNode(): switch to next node: no nodes left");
            return false;
        }

        // ensure that the last node is not null
        if (navigLastNode == null)
            navigLastNode = navigCurrentNode;

        // get next node distance
        int localDistance = (int) memory.self.getSpaceDistance (navigCurrentNode.location);

        // is this next node a mover?
        if (navigCurrentNode.UnrealID.indexOf ("LiftCenter") > -1)
        {
            // setup mover sequence
            navigStage = Stage.FirstMoverStage ();
            // setup local timeout for mover
            navigNodeTimeout = getTimeout (navigTimeout, localDistance + 400);
        }
        // are we still moving on mover?
        else if (navigStage.mover)
        {
            // setup local timeout for mover
            navigNodeTimeout = getTimeout (navigTimeout, localDistance + 400);
            // init the runner
            runner.initRunner ();
        }
        // no movers
        else
        {
            // setup local timeout based on distance
            navigNodeTimeout = getTimeout (navigTimeout, localDistance);
            // init the runner
            runner.initRunner ();
        }

        // switch to next node
        log.fine (
            "Navigator.switchToNextNode(): switch to next node " + navigCurrentNode.UnrealID
            + ", distance " + localDistance
            + ", timeout " + navigNodeTimeout
            + ", reachable " + navigCurrentNode.reachable
            + ", mover " + navigStage.mover
        );

        return true;
    }

    /*========================================================================*/

    /**
     * Tries to navigate the agent safely to the current navigation node.
     * @return Next stage of the navigation progress.
     */
    private Stage navigToCurrentNode ()
    {
        // get the distance from the current node
        int localDistance = (int) memory.self.getSpaceDistance (navigCurrentNode.location);
        // get the distance from the current node (neglecting jumps)
        int localDistance2 = (int) memory.self.getSpaceDistance(
            Triple.add(navigCurrentNode.location, new Triple (0,0,100))
        );

        // where are we going to run to
        Triple location = navigCurrentNode.location;
        // and what are we going to look at
        Triple focus = (navigNextNode == null) ? location : navigNextNode.location;

        // run to the current node..
        if (!runner.runToLocation (location, focus, navigCurrentNode.reachable))
        {
            log.fine ("Navigator.navigToCurrentNode(): navigation to current node failed");
            return Stage.CRASHED;
        }

        // we're still running
        log.finest ("Navigator.navigToCurrentNode(): traveling to current node: " + localDistance);

        // are we close enough to think about the next node?
        if ( (localDistance < 600) || (localDistance2 < 600) )
        {
            // prepare the next node only when it is not already prepared..
            if (navigCurrentNode == navigNextNode)
            {
                // prepare next node in the path
                prepareNextNode ();
            }
        }

        // are we close enough to switch to the next node?
        if ( (localDistance < 200) || (localDistance2 < 200) )
        {
            // switch navigation to the next node
            if (!switchToNextNode ())
            {
                // switch to the direct navigation
                log.fine ("Navigator.navigToCurrentNode(): switch to direct navigation");
                return initDirectly(navigDestination);
            }
        }

        // well, we're still running
        return navigStage;
    }

    /*========================================================================*/

    /**
     * Tries to navigate the agent safely along mover navigation nodes.
     *
     * <h4>Pogamut troubles</h4>
     *
     * Since the engine does not send enough reasonable info about movers and
     * their frames, the agent relies completely and only on the associated
     * navigation points. Fortunatelly, LiftCenter navigation points move with
     * movers.
     *
     * <p>Well, do not get too excited. Pogamut seems to update the position of
     * LiftCenter navpoint from time to time, but it's not frequent enough for
     * correct and precise reactions while leaving lifts.</p>
     *
     * @return Next stage of the navigation progress.
     */
    private Stage navigThroughMover ()
    {
        Stage stage = navigStage;

        // update positions of both navigation nodes since they may move
        // FUTURE: to be tested when navpoints are properly refreshed in memory
        navigLastNode = memory.items.getNavPoint (navigLastNode.UnrealID);
        navigCurrentNode = memory.items.getNavPoint (navigCurrentNode.UnrealID);

        // get horizontal distance from the mover center node
        int hDistance = (int) memory.self.getPlanarDistance (navigCurrentNode.location);
        // get vertical distance from the mover center node
        int vDistance = (int) memory.self.getVerticalDistance (navigCurrentNode.location);

        // wait for the current node to come close in both, vert and horiz
        // the horizontal distance can be quite long.. the agent will hop on
        if ( (vDistance > 50) || (hDistance > 500) )
        {
            // run to the last node, the one we're waiting on
            if (!runner.runToLocation (navigLastNode.location, navigCurrentNode.location, navigLastNode.reachable))
            {
                log.fine ("Navigator.navigThroughMoverWait("+stage+"): navigation to last node failed");
                return Stage.CRASHED;
            }
            // and keep waiting for the mover
            log.finer (
                "Navigator.navigThroughMoverWait("+stage+"): waiting for mover"
                + ", node " + navigCurrentNode.UnrealID
                + ", vDistance " + vDistance + ", hDistance " + hDistance
            );
            return navigStage;
        }

        // switch to the next mover step
        navigStage = navigStage.next ();
        log.finer (
            "Navigator.navigThroughMoverWait("+stage+"): mover arrived"
            + ", node " + navigCurrentNode.UnrealID
            + ", vDistance " + vDistance + ", hDistance " + hDistance
        );

        // first of all, switch to the next node
        if (!switchToNextNode ())
        {
            // switch to the direct navigation
            log.fine ("Navigator.navigThroughMoverMove("+stage+"): switch to direct navigation");
            return initDirectly (navigDestination);
        }

        // prepare next node ASAP
        prepareNextNode ();

        // and resolve the next mover step
        return navigStage.mover
            ? navigThroughMover ()
            : navigToCurrentNode ();
    }

    /*========================================================================*/

    /**
     * Tries to calculate an optimal timeout for traveling to distant location.
     * @param suggestedTimeout Timeout suggested by other intel. May be zero.
     * @param distance Distance, at which the traveling is to be experienced.
     * @return Number of loops of agent logic.
     */
    public int getTimeout (int suggestedTimeout, double distance)
    {
        // what's the ideal timeout for the distance? (with minimum of 2 secs)
        int timeout = (int) (Math.max (2, distance / 200) * main.logicFrequency);

        // do we have some upper timeout limit?
        if (suggestedTimeout > 0)
            return Math.min (timeout, suggestedTimeout);

        return timeout;
    }

    /*========================================================================*/

    /**
     * Enum of types of terminating navigation stages.
     */
    private enum TerminatingStageType {
        /** Terminating with success. */
        SUCCESS (false),
        /** Terminating with failure. */
        FAILURE (true);

        /** Whether the terminating with failure. */
        public boolean failure;

        /**
         * Constructor.
         * @param failure Whether the terminating with failure.
         */
        private TerminatingStageType (boolean failure)
        {
            this.failure = failure;
        }
    };

    /**
     * Enum of types of mover navigation stages.
     */
    private enum MoverStageType {
        /** Waiting for mover. */
        WAITING,
        /** Riding mover. */
        RIDING;
    };

    /**
     * All stages the navigation can come to.
     */
    public enum Stage
    {
        /**
         * Running directly to the destination.
         */
        REACHING ()
        {
            protected Stage next () { return this; }
        },
        /**
         * Navigating along the path.
         */
        NAVIGATING ()
        {
            protected Stage next () { return this; }
        },
        /**
         * Waiting for a mover to arrive.
         */
        AWAITING_MOVER (MoverStageType.WAITING)
        {
            protected Stage next () { return RIDING_MOVER; }
        },
        /**
         * Waiting for a mover to ferry.
         */
        RIDING_MOVER (MoverStageType.RIDING)
        {
            protected Stage next () { return NAVIGATING; }
        },
        /**
         * Navigation cancelled by outer force.
         */
        CANCELED (TerminatingStageType.FAILURE)
        {
            protected Stage next () { return this; }
        },
        /**
         * Navigation timeout reached.
         */
        TIMEOUT (TerminatingStageType.FAILURE)
        {
            protected Stage next () { return this; }
        },
        /**
         * Navigation failed because of troublesome obstacles.
         */
        CRASHED (TerminatingStageType.FAILURE)
        {
            protected Stage next () { return this; }
        },
        /**
         * Navigation finished sucessfully.
         */
        COMPLETED (TerminatingStageType.SUCCESS)
        {
            protected Stage next () { return this; }
        };

        /*====================================================================*/

        /**
         * Whether the navigation has failed.
         */
        private boolean mover;
        /**
         * Whether the nagivation is terminated.
         */
        public boolean terminated;
        /**
         * Whether the navigation has failed.
         */
        public boolean failure;

        /*====================================================================*/

        /**
         * Constructor: Not finished, not failed
         */
        private Stage ()
        {
            this.mover = false;
            this.terminated = false;
            this.failure = false;
        }

        /**
         * Constructor: mover.
         * @param type Type of mover navigation stage.
         */
        private Stage (MoverStageType type)
        {
            this.mover = true;
            this.terminated = false;
            this.failure = false;
        }

        /**
         * Constructor: terminating.
         * @param type Type of terminating navigation stage.
         */
        private Stage (TerminatingStageType type)
        {
            this.mover = false;
            this.terminated = true;
            this.failure = type.failure;
        }

        /*====================================================================*/

        /**
         * Retreives the next step of navigation sequence the stage belongs to.
         * @return The next step of navigation sequence. Note: Some stages are
         * not part of any logical navigation sequence. In such cases, this
         * method simply returns the same stage.
         */
        protected abstract Stage next ();

        /*====================================================================*/

        /**
         * Returns the first step of mover sequence.
         * @return The first step of mover sequence.
         */
        protected static Stage FirstMoverStage ()
        {
            return AWAITING_MOVER;
        }
    }

    /*========================================================================*/

    /**
     * Listening class for messages from engine.
     */
    private class Listener extends LoqueListener
    {
        /**
         * Agent received reachability response.
         * @param msg Message to handle.
         */
        private void msgReachable (Reachable msg)
        {
            // parse the response id
            int ID = Integer.parseInt(msg.pongID);
            synchronized (navigPrepareRequest)
            {
                // is is the correct location?
                if (navigPrepareRequest.ID == ID)
                {
                    // process this request
                    navigNextNode.reachable = msg.reachable;
                    log.finer ("Navigator.Listener: next node reachability: " + msg.reachable);
                }
            }
        }

        /**
         * Message switch.
         * @param msg Message to handle.
         */
        protected void processMessage (MessageObject msg)
        {
            switch (msg.type)
            {
                case REACHABLE:
                    msgReachable ((Reachable) msg);
                    return;
            }
        }

        /**
         * Constructor: Signs up for listening.
         */
        private Listener ()
        {
            body.addTypedRcvMsgListener(this, MessageType.REACHABLE);
        }
    }

    /** Listener. */
    private LoqueListener listener;

    /*========================================================================*/

    /**
     * Loque Runner.
     */
    private LoqueRunner runner;

    /*========================================================================*/

    /** Agent's main. */
    protected Main main;
    /** Loque memory. */
    protected LoqueMemory memory;
    /** Agent's body. */
    protected AgentBody body;
    /** Agent's log. */
    protected Logger log;

    /*========================================================================*/

    /**
     * Constructor.
     * @param main Agent's main.
     * @param memory Loque memory.
     */
    public LoqueNavigator (Main main, LoqueMemory memory)
    {
        // setup reference to agent
        this.main = main;
        this.memory = memory;
        this.body = main.getBody ();
        this.log = main.getLogger ();

        // create runner object
        this.runner = new LoqueRunner (main, memory);

        // create listener
        this.listener = new Listener ();
    }
}